/** 
 * JSONSchema Validator - Validates JavaScript objects using JSON Schemas 
 *	(http://www.json.com/json-schema-proposal/)
 *
 * Copyright (c) 2007 Kris Zyp SitePen (www.sitepen.com)
 * Licensed under the MIT (MIT-LICENSE.txt) license.
To use the validator call JSONSchema.validate with an instance object and an optional schema object.
If a schema is provided, it will be used to validate. If the instance object refers to a schema (self-validating), 
that schema will be used to validate and the schema parameter is not necessary (if both exist, 
both validations will occur). 
The validate method will return an array of validation errors. If there are no errors, then an 
empty list will be returned. A validation error will have two properties: 
"property" which indicates which property had the error
"message" which indicates what the error was
 */

/* 
	** extendet version **
	Der Schema-Validator wird dahingehend erweitert, dass der die wichtigsten Eigenschaften zur Erzeugung
	von Forms in einem Array speichert.
	Folgende Eigenschaften werden gespeichert:
	- name			// Objektname, fuer Strukturierung
	- key			// Schluesselwort
	- value			// Wert
	- form			// Art der Form (simple / list / radio),
				// simple = einfaches Feld,  list = ComboBox, radio = Radiobuttons
	- label			// bei form = list: 	sichtbarer Eintrag in der Liste
	- listVal		// bei form = list:	ausgewaehlter Wert aus der Liste
	- validVal		// Voreinstellung gueltiger Werte
	- readOnly		// Wert soll nur lesbar sein
	- maxLength		// Wert darf nur max. Anzahl von angegebenen Zeichen haben
	- maximum		// Wert darf angegebenes Maximum nicht ueberschreiten (R.S. 09.02.11)
	- element		// Objektpfad
	- valtype		// Type des values (string, number, ...)
	- hidden		// true = Element wird nicht dargstellt
 
	05/2009 R.Stiawa
 */ 
var formProp = new Array();
var jsonschema_debug = 0;		// Debug-Ausgaben 
 
 
 
JSONSchema = 
{
	validate : function(/*Any*/instance,/*Object*/schema) 
	{
		// Summary:
		//  	To use the validator call JSONSchema.validate with an instance object and an optional schema object.
		// 		If a schema is provided, it will be used to validate. If the instance object refers to a schema (self-validating), 
		// 		that schema will be used to validate and the schema parameter is not necessary (if both exist, 
		// 		both validations will occur). 
		// 		The validate method will return an object with two properties:
		// 			valid: A boolean indicating if the instance is valid by the schema
		// 			errors: An array of validation errors. If there are no errors, then an 
		// 					empty list will be returned. A validation error will have two properties: 
		// 						property: which indicates which property had the error
		// 						message: which indicates what the error was
		//
		
		// ========================================================================================
		// Insert 05/2009 R.St.
		// clear Array
		formProp.length = 0;
		// ========================================================================================
		
		return this._validate(instance,schema,false);
	},
	
	checkPropertyChange : function(/*Any*/value,/*Object*/schema, /*String*/ property) 
	{
		// Summary:
		// 		The checkPropertyChange method will check to see if an value can legally be in property with the given schema
		// 		This is slightly different than the validate method in that it will fail if the schema is readonly and it will
		// 		not check for self-validation, it is assumed that the passed in value is already internally valid.  
		// 		The checkPropertyChange method will return the same object type as validate, see JSONSchema.validate for 
		// 		information.
		//
		return this._validate(value,schema, property || "property");
	},
	
	_validate : function(/*Any*/instance,/*Object*/schema,/*Boolean*/ _changing) 
	{
		var errors = [];
		// validate a value against a property definition
	
		// ========================================================================================
		// Insert 05/2009 R.St.
		// Get an array of properties for building forms
		function getFormProperties(value, propDef, path, object)
		{
			// alert ("object = "+value+"  "+propDef+"  "+path+"  "+object);

			if ((typeof value == "object") &&
				!(value instanceof Array))
			{
				// we got an object, but not an array
				return;
			}

			if (value instanceof Array)
			{
				// Array nur als Werte-Array auswerten, nicht wenn es Objekte enthaelt;
				// Wenn sie Objekte enthaelt, wird die Funktion rekursiv aufgerufen
				if (typeof value[0] == 'object')		
					return;
			}			
			
			var len = formProp.length;
			formProp[len] = new Object();			// new Object inside array

			// element (object path)
			var element;
			if (path == "")		element	= object;
			else				element = path+"."+object;		
			if (jsonschema_debug)	alert (element);
			formProp[len].element = element;		// object path
			
			if (!(value instanceof Array))
				if (jsonschema_debug)	alert ("value = "+value+"   "+typeof(value));

			formProp[len].key = object;							// key
			formProp[len].value = value;						// value
			formProp[len].valtype = typeof(value);				// type of value (string, number ..)
						
			if (propDef['options'])
			{
				formProp[len].form = "list";					// form
				var opt = propDef['options'];
				var l = opt.length;

				formProp[len].label = new Array();
				formProp[len].listVal = new Array();
				for (var j=0; j<l; j++)
				{
					formProp[len].label[j] = opt[j].label;		// label
					formProp[len].listVal[j] = opt[j].value;	// listVal
				}				
			}	
			else
			{
				if (value instanceof Array)
					; 	// Objekt - Element waere ein Array 		
				else
					formProp[len].form = "simple";				// form
			}
			
			if (propDef['enum'])
			{
				formProp[len].form = "radio";					//form	
				var valid = propDef['enum'];
				var l = valid.length;

				formProp[len].validVal = new Array();
				for (var j=0; j<l; j++)					
				{
					formProp[len].validVal[j] = valid[j];		// validVal
				}
			}
		
			if (propDef['readonly'])
			{
				if (jsonschema_debug)	alert ("readonly");
				formProp[len].readOnly = true;					// readonly
			}
		
			if (propDef['maximum'])
			{
				if (jsonschema_debug)	alert ("maximum");
				var maximum = propDef['maximum'];
				formProp[len].maximum = maximum;				// maximum
			}

			if (propDef['maxLength'])
			{
				if (jsonschema_debug)	alert ("maxLength");
				var maxLength = propDef['maxLength'];
				formProp[len].maxLength = maxLength;			// maxLength
			}
			
			if (propDef['hidden'])
			{
				if (jsonschema_debug)	alert ("hidden");
				var hidden = propDef['hidden'];
				formProp[len].hidden = hidden;					// hidden
			}

		}

		// ========================================================================================

		function checkProp(value, schema, path,i)
		{

			var l;
			path += path ? typeof i == 'number' ? '[' + i + ']' : typeof i == 'undefined' ? '' : '.' + i : i;
			function addError(message)
			{
				errors.push({property:path,message:message});
			}
		
			// ========================================================================================
			// Insert 05/2009 R.St.
			if (typeof value == 'object')
			{
				if (path != '')
				{
					if (jsonschema_debug)	alert ("Object\nName: "+path);
					var len = formProp.length;
					formProp[len] = new Object();		// new Object inside array
					formProp[len].name = path;			// name for new section			
				}
			}
			// ========================================================================================
		
			if ((typeof schema != 'object' || schema instanceof Array) && 
				(path || typeof schema != 'function'))
			{
				if(typeof schema == 'function')
				{
					if (!(value instanceof schema))
					{
						addError("is not an instance of the class/constructor " + schema.name);
					}
				}
				else if (schema)
				{
					addError("Invalid schema/property definition " + schema);
				}
				return null;
			}
			
			if (_changing && schema.readonly)
			{
				addError("is a readonly field, it can not be changed");
			}
			if (schema['extends'])
			{ 
				// if it extends another schema, it must pass that schema as well
				checkProp(value,schema['extends'],path,i);
			}

			// validate a value against a type definition
			function checkType(type,value)
			{
				if (type)
				{
					if (typeof type == 'string' && type != 'any' && 
						(type == 'null' ? value !== null : typeof value != type) && 
						!(value instanceof Array && type == 'array') &&
						!(type == 'integer' && value%1===0))
					{
						return [{property:path,message:(typeof value) + " value found, but a " + type + " is required"}];
					}
					
					if (type instanceof Array)
					{
						var unionErrors=[];
						for (var j = 0; j < type.length; j++)
						{ 	// a union type 
							if(!(unionErrors=checkType(type[j],value)).length)
							{
								break;
							}
						}
						if (unionErrors.length)
						{
							return unionErrors;
						}
					}
					else if (typeof type == 'object')
					{
						var priorErrors = errors;
						errors = []; 
						checkProp(value,type,path);
						var theseErrors = errors;
						errors = priorErrors;
						return theseErrors; 
					} 
				}
				return [];
			}
		
			if (value === undefined)
			{
				if (!schema.optional)
				{  
					addError("is missing and it is not optional");
				}
			}
			else
			{
				errors = errors.concat(checkType(schema.type,value));
				if (schema.disallow && !checkType(schema.disallow,value).length)
				{
					addError(" disallowed value was matched");
				}
				if (value !== null)
				{
					if (value instanceof Array)
					{
						if (schema.items)
						{
							if (schema.items instanceof Array)
							{
								for (i=0,l=value.length; i<l; i++)
								{
									errors.concat(checkProp(value[i],schema.items[i],path,i));
								}
							}
							else
							{
								for (i=0,l=value.length; i<l; i++)
								{
									errors.concat(checkProp(value[i],schema.items,path,i));
								}
							}							
						}
						if (schema.minItems && value.length < schema.minItems)
						{
							addError("There must be a minimum of " + schema.minItems + " in the array");
						}
						if (schema.maxItems && value.length > schema.maxItems)
						{
							addError("There must be a maximum of " + schema.maxItems + " in the array");
						}
					}
					else if (schema.properties)
					{
						errors.concat(checkObj(value,schema.properties,path,schema.additionalProperties));
					}
					if (schema.pattern && typeof value == 'string' && !value.match(schema.pattern))
					{
						addError("does not match the regex pattern " + schema.pattern);
					}
					if (schema.maxLength && typeof value == 'string' && value.length > schema.maxLength)
					{
						addError("may only be " + schema.maxLength + " characters long");
					}
					if (schema.minLength && typeof value == 'string' && value.length < schema.minLength)
					{
						addError("must be at least " + schema.minLength + " characters long");
					}
					if (typeof schema.minimum !== undefined && typeof value == typeof schema.minimum && 
						schema.minimum > value)
					{
						addError("must have a minimum value of " + schema.minimum);
					}
					if (typeof schema.maximum !== undefined && typeof value == typeof schema.maximum && 
						schema.maximum < value)
					{
						addError("must have a maximum value of " + schema.maximum);
					}
					if (schema['enum'])
					{
						var enumer = schema['enum'];
						l = enumer.length;
						var found;
						for (var j = 0; j < l; j++)
						{
							if (enumer[j]===value)
							{
								found=1;
								break;
							}
						}
						if (!found)
						{
							addError("does not have a value in the enumeration " + enumer.join(", "));
						}
					}
					if (typeof schema.maxDecimal == 'number' && 
						(value.toString().match(new RegExp("\\.[0-9]{" + (schema.maxDecimal + 1) + ",}"))))
					{
						addError("may only have " + schema.maxDecimal + " digits of decimal places");
					}
				}
			}

			return null;
		}	// end checkProp()

		// validate an object against a schema
		function checkObj(instance,objTypeDef,path,additionalProp)
		{

			if(typeof objTypeDef =='object')
			{
				if (typeof instance != 'object' || instance instanceof Array)
				{
					errors.push({property:path,message:"an object is required"});
				}
				
				for (var i in objTypeDef)
				{ 
					if (objTypeDef.hasOwnProperty(i) && !(i.charAt(0) == '_' && i.charAt(1) == '_'))
					{
						var value = instance[i];
						var propDef = objTypeDef[i];
						checkProp(value,propDef,path,i);
						
						// ========================================================================================
						// Insert 05/2009 R.St.
						getFormProperties (value, propDef, path, i);
						// ========================================================================================
					}
				}
			}
			for (i in instance)
			{
				if (instance.hasOwnProperty(i) && !(i.charAt(0) == '_' && i.charAt(1) == '_') && 
					objTypeDef && !objTypeDef[i] && additionalProp===false)
				{
					errors.push({property:path,message:(typeof value) + "The property " + i +
							" is not defined in the schema and the schema does not allow additional properties"});
				}
				var requires = objTypeDef && objTypeDef[i] && objTypeDef[i].requires;
				if (requires && !(requires in instance))
				{
					errors.push({property:path,message:"the presence of the property " + i + " requires that " + 
								requires + " also be present"});
				}
				value = instance[i];
				if (objTypeDef && typeof objTypeDef == 'object' && !(i in objTypeDef))
				{
					checkProp(value,additionalProp,path,i); 
				}
				if (!_changing && value && value.$schema)
				{
					errors = errors.concat(checkProp(value,value.$schema,path,i));
				}
			}

			return errors;
		}	// end checkObj()
	
		if (schema)
			checkProp(instance,schema,'',_changing || '');

		if (!_changing && instance && instance.$schema)
			checkProp(instance,instance.$schema,'','');

		return {valid:!errors.length,errors:errors};

	}	// end validate()
}

// ************************************************************************************************
// ************************************************************************************************

